﻿//
// 3D物理エンジン Open Dynamics Engine を用いた一輪車のプログラム
//

#include "main.h" // 全てのc/cppコードで共通に参照するヘッダーファイル
#include <windows.h>

void InitializeAllParameter(void)
{
	int i;
	InitALL();
	srand(time(NULL));
	switch(RunFlag)
	{
		case(SingleWheelTypeModel):
			Dimension = 4; // 状態の次数
			NumberOfInput = 1; // 入力数
			break;
		case(InertiaRotorPendulum):
			Dimension = 4;
			NumberOfInput = 1;
			break;
		default:
			break;
	}

	InertiaRotorAngle = 0.0; // 慣性ローターのローターの回転角

	LearningLength = 1; // 現在学習回数(クロック)
	LearningLengthFinal = 1; // 最終学習回数(クロック) 初期値1
	BlockLength = 1; // 現在のブロック長さ(ブロック)
	BlockLengthFinal = (Dimension + NumberOfInput) * 10; // 最終ブロック長さ(ブロック)
	RunTime = 0; // 制御の実時間 [msec]
	Delta = 0.000001; // 逆行列補題の初期値 δ あまり小さい値にすると制御する時に学習に失敗する恐れがある。
	Sigma = 50.0; // 追加入力雑音の振幅 σ
	Lambda1 = 1.0; // λ1 [0<=1] 学習係数
	Lambda2 = 1.0; // λ2 [0<=1] 入力係数
	C.Initialize(Dimension, Dimension, 0.0); // 状態に対する重み行列 C の初期化
	C.ZeroClear(); // 念のため行列を0.0でクリア
	D.Initialize(NumberOfInput, NumberOfInput, 0.0); // 入力に対する重み行列 D の初期化
	D.ZeroClear(); // 念のため行列を0.0でクリア
	G.Initialize(NumberOfInput, Dimension, 0.0); // フィードバックゲイン G の行列を初期化
	G.ZeroClear(); // 念のため行列を0.0でクリア
	H.Initialize(NumberOfInput, NumberOfInput, 0.0); // 学習のための統計処理を含む H 行列を初期化
	H.ZeroClear(); // 念のため行列を0.0でクリア
	x.Initialize(Dimension, 1, 0.0); // 状態 x を初期化
	x.ZeroClear(); // 念のため行列を0.0でクリア
	v.Initialize(NumberOfInput, 1, 0.0); // 追加入力雑音 v を初期化
	v.ZeroClear(); // 念のため行列を0.0でクリア
	u.Initialize(NumberOfInput, 1, 0.0); // 入力 u を初期化
	u.ZeroClear(); // 念のため行列を0.0でクリア
	z.Initialize(Dimension + NumberOfInput, 1, 0.0); // 応答信号 z の初期化
	z.ZeroClear(); // 念のため行列を0.0でクリア
	xi.Initialize(Dimension + NumberOfInput, 1, 0.0); // 識別信号 ξ の初期化
	xi.ZeroClear(); // 念のため行列を0.0でクリア
	P.Initialize(Dimension, Dimension, 0.0); // リカッチ方程式の解 P の初期化
	P.ZeroClear(); // 念のため行列を0.0でクリア
	Psi.Initialize(Dimension + NumberOfInput, Dimension + NumberOfInput, 0.0); // Ψの初期化
	Psi.ZeroClear(); // 念のため行列を0.0でクリア
	Upsilon.Initialize(Dimension, Dimension + NumberOfInput, 0.0); // Υの初期化
	Upsilon.ZeroClear(); // 念のため行列を0.0でクリア
	Xi.Initialize(Dimension + NumberOfInput, Dimension + NumberOfInput, 0.0); // Ξの初期化
	Xi.ZeroClear(); // 念のため行列を0.0でクリア
	Gamma.Initialize(Dimension + NumberOfInput, Dimension + NumberOfInput, 0.0); // Γの初期化
	Gamma.ZeroClear(); // 念のため行列を0.0でクリア
	Gamma_aa.Initialize(Dimension, Dimension, 0.0); // Γaaの初期化
	Gamma_aa.ZeroClear(); // 念のため行列を0.0でクリア
	Gamma_ab.Initialize(Dimension, NumberOfInput, 0.0); // Γabの初期化
	Gamma_ab.ZeroClear(); // 念のため行列を0.0でクリア
	Gamma_ba.Initialize(NumberOfInput, Dimension, 0.0); // Γbaの初期化
	Gamma_ba.ZeroClear(); // 念のため行列を0.0でクリア
	Gamma_bb.Initialize(NumberOfInput, NumberOfInput, 0.0); // Γbbの初期化
	Gamma_bb.ZeroClear(); // 念のため行列を0.0でクリア
	I.LoadIdentity(Dimension + NumberOfInput); // 単位行列Iを作成

	phystime = 0.0; // 物理エンジン内部の時間

	// 状態の重みを設定
	for(i = 0; i < Dimension; i++)
	{
		C(i, i) = 1.0;
	}

	// 入力の重みを設定
	for(i = 0; i < NumberOfInput; i++)
	{
		D(i, i) = 1.0;
	}

	// 学習制御と適応制御に関連するパラメータ
	for(i = 0; i < NumberOfInput; i++)
	{
		H(i, i) = 1.0;
	}
}

void zUpdate(void)
{
	int i;
	CMatrix z_a;
	CMatrix z_b;
	
	z_a.Initialize(Dimension, 1, 0.0);
	z_b.Initialize(NumberOfInput, 1, 0.0);
	// z_a z_b を一つの応答信号 z とする
	z_a = C * x; // C * x 英語論文(2)式
	z_b = D * u; // D * u 英語論文(2)式
	for(i = 0; i < Dimension; i++)
	{
		z(i, 0) = z_a(i, 0); // C * x 英語論文(2)式
	}
	for(i = Dimension; i < Dimension + NumberOfInput; i++)
	{
		z(i, 0) = z_b(i - Dimension, 0); // D * u 英語論文(2)式
	}
}

void xiUpdate(void)
{
	int i;

	// 識別信号 ξ を生成 ξ_a OR ξ_b = ξ
	for(i = 0; i < Dimension; i++)
	{
		xi(i, 0) = x(i, 0); // ξ_a = x 感度計算のための識別信号構成 英語論文(11)式
	}
	for(i = Dimension; i < Dimension + NumberOfInput; i++)
	{
		xi(i, 0) = v(i - Dimension, 0); // ξ_b = v 感度計算のための識別信号構成 英語論文(11)式
	} // ξ_a OR ξ_b = ξ
}

// Record Real Time and state(=x,=sensor value)
void RecordGraphx(void)
{
	int i;
	state_file << RunTime << " ";
	for(i = 0; i < Dimension; i++)
	{
		state_file << x(i, 0) << " ";
	}
	state_file << endl;
	state_csv_file << RunTime << ",";
	for(i = 0; i < Dimension-1; i++)
	{
		state_csv_file << x(i, 0) << ",";
	}
	state_csv_file << x(Dimension-1, 0);
	state_csv_file << endl;
}

void RecordGraphrpm(void)
{
	int i;
	rpm_file << RunTime << " ";
	rpm_file << ((dJointGetHingeAngleRate(InertiaRotorPendulumJoint[2]) * DT) / (2 * M_PI)) * 60.0;
	rpm_file << endl;
}

// Record Learning Length and Feedback gain(=k)
void RecordGraphG(void)
{
	int i, j;
	gain_file << LearningLength << " ";
	for(i = 0; i < NumberOfInput; i++)
	{
		for(j = 0; j < Dimension; j++)
		{
			gain_file << G(i, j) << " ";
		}
	}
	gain_file << endl;
	gain_csv_file << LearningLength << ",";
	for(i = 0; i < NumberOfInput; i++)
	{
		for(j = 0; j < Dimension; j++)
		{
			gain_csv_file << G(i, j) << ",";
			
		}
	}
	//gain_csv_file << G(NumberOfInput-1, Dimension-1);
	gain_csv_file << endl;
}

// Record Real Time and input value(=voltage, motor torque ...)
void RecordGraphu(void)
{
	int i;
	input_file << RunTime << " ";
	for(i = 0; i < NumberOfInput; i++)
	{
		input_file << u(i, 0) << " ";
	}
	input_file << endl;
	input_csv_file << RunTime << ",";
	for(i = 0; i < NumberOfInput-1; i++)
	{
		input_csv_file << u(i, 0) << ",";
	}
	input_csv_file << u(NumberOfInput-1, 0);
	input_csv_file << endl;
}

void Control(void)
{
	int i;

	//
	// 追加入力雑音の生成
	//
	for(i = 0; i < NumberOfInput; i++)
	{
		v(i, 0) = Sigma * ((((double)rand() / (double) RAND_MAX) * (1.9999999999999999999999999)) - 1.0); // [-1, +1]の振幅を持つ追加入力雑音を生成
		// 生成された雑音は線形独立
	}
	

	//
	// 線形状態フィードバック制御 英語論文(9)式 日本語資料(49)式
	//
	switch(ControlFlag)
	{
		case(LearningControl):
			u = G * x + v; // u = G * x + v 英語論文(9)式
			break;
		case(InitialValueResponse):
			u = G * x; // u = G * x (雑音を除く)
			break;
		case(AdaptiveControl):
			u = G * x + v; // u = G * x + v 英語論文(9)式
			break;
		default:
			break;
	}
	RecordGraphu();
	

	//
	// 応答信号 z の生成 英語論文(2)式 日本語資料(80)式
	//
	zUpdate(); // 応答信号 z を更新する z = (C * x, D * u)^T 英語論文(2)式
	

	//
	// 識別信号ξの生成
	//
	xiUpdate(); // 識別信号 ξ を更新する ξ = (x, v)^T 英語論文(11)式
	
}

void SensitivityMain(void)
{
		//
		// ブロックの開始時での各パラメータの初期化
		//
		if(BlockLength == 1)
		{ // ブロックの長さが1なら
			
			//
			// Ψのブロック毎の初期化
			//
			Psi.ZeroClear(); // Ψをゼロクリア
			

			//
			// Υのブロック毎の初期化
			//
			Upsilon.ZeroClear(); // Υをゼロクリア

			//
			// Ξ のブロック毎の初期化
			//		
			Xi.ZeroClear(); // Ξをゼロクリア
			Xi = (I * (1.0 / Delta)); // 逆行列補題の初期値 1/δ * I 英語論文(21)式

			//
			// Γのブロック毎の初期化
			//
			Gamma.ZeroClear(); // Γをゼロクリア
			Gamma_aa = P; // Γaaの初期値にPを代入
			Gamma_ab.ZeroClear(); // Γabをゼロクリア
			Gamma_ba.ZeroClear(); // Γbaをゼロクリア
			Gamma_bb = (D.Transpose()) * D; // Γbbの初期値に D^T * D を代入
			GammaUpdateFromGamma_aa_ab_ba_bb(); // Γaa,Γab,Γba,ΓbbからΓを更新
		}

		//
		// Ξ の計算
		//
		// Ξ(t+1) = Ξ(t) - (Ξ * ξ) * (ξ^T * Ξ * ξ + 1)^(-1) * (ξ^T * Ξ) の計算 英語論文(24)式
		Xi = Xi - ((Xi * xi) * (((xi.Transpose() * (Xi * xi)) + 1.0).InverseLUD()) * (xi.Transpose() * Xi));
		

		//
		// 感度行列 Γ の計算
		//
		// Γ(t+1) = Γ(t)
		//            + Ξ * ξ * ((z^T * Ψ + x^T * Υ) * Ξ - ξ^T * Γ)
		//            + (Ξ * (Υ^T * x + Ψ^T * z) - Γ * ξ) * ξ^T * Ξ
		//            + Ξ * ξ * (z^T * z + x^T * P * x + ξ^T * Γ * ξ) * ξ^T * Ξ の計算 英語論文(25)式
		Gamma = Gamma
			+ (Xi * xi) * ((((z.Transpose() * Psi) + (x.Transpose() * Upsilon)) * Xi) - (xi.Transpose() * Gamma))
			+ (Xi * ((Psi.Transpose() * z) + (Upsilon.Transpose() * x)) - (Gamma * xi)) * xi.Transpose() * Xi
			+ (Xi * xi) * ((z.Transpose() * z) + (x.Transpose() * P * x) + (xi.Transpose() * Gamma * xi)) * (xi.Transpose() * Xi);
		Gamma_aa_ab_ba_bbUpdateFromGamma(); // ΓからΓaa,Γab,Γba,Γbbを更新
		

		//
		// 感度 Ψ(t+1)の計算 英語論文(22)式
		//
		Psi = (Psi + (z * xi.Transpose())); // Ψ(t+1) = Ψ(t) + z * ξ^T 英語論文(22)式
		

		//
		// 感度 Υ(t+1)の計算 英語論文(23)式
		//
		Upsilon = (Upsilon + (P * (x * xi.Transpose()))); // Υ(t+1) = Υ(t) + P * x * ξ^T 英語論文(23)式
		
}

void GainUpdate(void)
{
	//
	// ゲインの更新 G
	//
	G = G - (((Gamma_bb + H).InverseLUD()) * Gamma_ba); // G(k+1) = G - (Γbb + H)^(-1) * Γba 英語論文(13)式
//	PrintMatrix("G = ", G);
	

	//
	// リカッチ方程式の解 P
	//
	// P(k+1) = Γaa - Γab * (Γbb + H)^(-1) * (Γbb + 2 * H) * (Γbb + H)^(-1) * (Γba) 英語論文(14)式
	P = Gamma_aa - (Gamma_ab * ((Gamma_bb + H).InverseLUD()) * (Gamma_bb + (H * 2.0)) * ((Gamma_bb + H).InverseLUD()) * Gamma_ba);
	

	//
	// Γab,Γbaの更新
	// 
	//  Γab = Γab + Γbb * {(Γbb + H)^(-1) * Γba} → (Γab = Γab + Γbb * {⊿Gk})
	/*
	Gamma_ab = Gamma_ab + Gamma_bb * (((Gamma_bb + H).InverseLUD()) * Gamma_ba); // Γab = Γab + Γbb * {(Γbb + H)^(-1) * Γba}
	Gamma_ba = Gamma_ab.Transpose(); // Γab^T = Γba、Γab = Γba^T
	Gamma_aa_ab_ba_bbUpdateFromGamma(); // ΓからΓaa,Γab,Γba,Γbbを更新
	*/

	//
	// 学習に関する統計処理のための計算 H
	//
	H = (H * Lambda1) + (Gamma_bb * Lambda2); // H(k+1) = λ1 * H + λ2 * Γbb 英語論文(15)式
}

void GammaUpdateFromGamma_aa_ab_ba_bb(void)
{
	int i;
	int j;

	// Γaa を Γ に適用
	for(i = 0; i < Dimension; i++)
	{
			for(j = 0; j < Dimension; j++)
			{
				Gamma(i, j) = Gamma_aa(i, j);
			}
	}

	// Γab を Γ に適用
	for(i = 0; i < Dimension; i++)
	{
		for(j = Dimension; j < Dimension + NumberOfInput; j++)
		{
			Gamma(i, j) = Gamma_ab(i, j - Dimension);
		}
	}

	// Γba を Γ に適用
	for(i = Dimension; i < Dimension + NumberOfInput; i++)
	{
		for(j = 0; j < Dimension; j++)
		{
			Gamma(i, j) = Gamma_ba(i - Dimension, j);
		}
	}

	// Γbb を Γ に適用
	for(i = Dimension; i < Dimension + NumberOfInput; i++)
	{
		for(j = Dimension; j < Dimension + NumberOfInput; j++)
		{
			Gamma(i, j) = Gamma_bb(i - Dimension, j - Dimension);
		}
	}
}

void Gamma_aa_ab_ba_bbUpdateFromGamma(void)
{
	int i;
	int j;

	// Γaa を Γ に適用
	for(i = 0; i < Dimension; i++)
	{
		for(j = 0; j < Dimension; j++)
		{
			Gamma_aa(i, j) = Gamma(i, j);
		}
	}

	// Γab を Γ に適用
	for(i = 0; i < Dimension; i++)
	{
		for(j = Dimension; j < Dimension + NumberOfInput; j++)
		{
			Gamma_ab(i, j - Dimension) = Gamma(i, j);
		}
	}

	// Γba を Γ に適用
	for(i = Dimension; i < Dimension + NumberOfInput; i++)
	{
		for(j = 0; j < Dimension; j++)
		{
			Gamma_ba(i - Dimension, j) = Gamma(i, j);
		}
	}

	// Γbb を Γ に適用
	for(i = Dimension; i < Dimension + NumberOfInput; i++)
	{
		for(j = Dimension; j < Dimension + NumberOfInput; j++)
		{
			Gamma_bb(i - Dimension, j - Dimension) = Gamma(i, j);
		}
	}
}

void ResetModel(void)
{
	switch(RunFlag)
	{
		case(SingleWheelTypeModel):
			// 状態リセット
			if ((x(0, 0) < -0.6) 
				|| (x(0, 0) > 0.6)
				|| (x(2, 0) < -0.8)
				|| (x(2, 0) > 0.8))
			{
			state_file << "# reset of state wheel and frame" << endl;
			restart();
			u.ZeroClear();
			x.ZeroClear();
			v.ZeroClear();
		
			wheel_rpy[1] = 0.0;
			prev_wheel_pitch = 0.0;
			frame_rpy[1] = 0.0;
			prev_frame_pitch = 0.0;
			printf("reset state!\n");
			}
			break;
		case(InertiaRotorPendulum):
			if ((x(0, 0) < -0.2) 
				|| (x(0, 0) > 0.2))
			{
				state_file << "# reset of state" << endl;
				restart();
				InertiaRotorAngle = 0.0;
				u.ZeroClear();
				x.ZeroClear();
				v.ZeroClear();
				printf("reset state!\n");
			}
			break;
		default:
			break;
	}
}


//
// 1輪車の車輪を作成
//
void SingleWheelTypeModelCreateWheel()
{
	dMass dmass;

	wheel.body = dBodyCreate(world); // 車輪を作成
	dMassSetZero(&(dmass)); // 車輪の質量，重心位置，慣性モーメントなどが入っている質量パラメータを初期化します
	dMassSetCylinderTotal(&(dmass), WHEEL_MASS, 3, WHEEL_RADIUS, WHEEL_LENGTH); // 車輪の物理的なサイズを決定
	dBodySetMass(wheel.body, &(dmass)); // 車輪の質量をボディに関連付け
	wheel.geom = dCreateCylinder(space, WHEEL_RADIUS, WHEEL_LENGTH); // 車輪の幾何情報を生成
	dGeomSetBody(wheel.geom, wheel.body); // 台の幾何情報と存在の一致
	
	// 車輪の初期角度を指定
	dMatrix3 R; // 回転行列
	dRFromAxisAndAngle(R, 1.0, 0.0, 0.0, (90.0 * M_PI) / 180.0); // x軸を中心として90°左に回転
	dBodySetRotation(wheel.body, R); // 回転行列Rを車輪に適用

	dBodySetPosition(wheel.body, 0.0, 0.0, WHEEL_RADIUS); // 初期位置を指定

	// 剛体の時間ステップでのモードを指定する
	// flagに1を設定して有限(finite)方位で更新する。計算負荷は大きいが正確な高速回転となる
	dBodySetFiniteRotationMode(wheel.body, 1);
}

//
// 1輪車のフレームを作成
//
void SingleWheelTypeModelCreateFrame()
{
	dMass dmass;

	frame.body = dBodyCreate(world); // フレームを作成
	dMassSetZero(&(dmass)); // フレームの質量，重心位置，慣性モーメントなどが入っている質量パラメータを初期化します．
	dMassSetBoxTotal(&(dmass), FRAME_MASS, FRAME_X, FRAME_Y, FRAME_Z); // フレームの物理的なサイズを決定 2番目の引数は密度
	dBodySetMass(frame.body, &(dmass)); // 質量をボディに適用
	dBodySetPosition(frame.body, 0.0, 0.0, (((FRAME_Z / 2.0) + WHEEL_RADIUS) - (FRAME_Z / 11.0))); // フレームの初期位置
	frame.geom = dCreateBox(space, FRAME_X, FRAME_Y, FRAME_Z); // フレームの幾何情報を生成
	dGeomSetBody(frame.geom, frame.body); // フレームの幾何情報と存在の一致

	// 剛体の時間ステップでのモードを指定する
	dBodySetFiniteRotationMode(frame.body, 1);
	// flagに1を設定して有限(finite)方位で更新する。計算負荷は大きいが正確な高速回転となる
}

// 慣性ロータを作成する関数
void InertiaRotorPendulumCreateRotor()
{
	dMass dmass;

	rotor.body = dBodyCreate(world); // 車輪を作成
	dMassSetZero(&(dmass)); // 車輪の質量，重心位置，慣性モーメントなどが入っている質量パラメータを初期化します
	dMassSetCylinderTotal(&(dmass), ROTOR_MASS, 3, ROTOR_RADIUS, ROTOR_LENGTH); // 車輪の物理的なサイズを決定
	dBodySetMass(rotor.body, &(dmass)); // 車輪の質量をボディに関連付け
	rotor.geom = dCreateCylinder(space, ROTOR_RADIUS, ROTOR_LENGTH); // 車輪の幾何情報を生成
	dGeomSetBody(rotor.geom, rotor.body); // 台の幾何情報と存在の一致
	
	// 車輪の初期角度を指定
	dMatrix3 R; // 回転行列
	dRFromAxisAndAngle(R, 0.0, 1.0, 0.0, (90.0 * M_PI) / 180.0); // y軸を中心として90°左に回転
	dBodySetRotation(rotor.body, R); // 回転行列Rを車輪に適用

	dBodySetPosition(rotor.body, 0.0, 0.0, PENDULUM_Z); // 初期位置を指定

	// 剛体の時間ステップでのモードを指定する
	// flagに1を設定して有限(finite)方位で更新する。計算負荷は大きいが正確な高速回転となる
	dBodySetFiniteRotationMode(rotor.body, 1);
}

// 慣性ロータの振り子を作成する関数
void InertiaRotorPendulumCreatePendulum()
{
	dMass dmass;

	pendulum.body = dBodyCreate(world); // フレームを作成
	dMassSetZero(&(dmass)); // フレームの質量，重心位置，慣性モーメントなどが入っている質量パラメータを初期化します．
	dMassSetBoxTotal(&(dmass), PENDULUM_MASS, PENDULUM_X, PENDULUM_Y, PENDULUM_Z); // フレームの物理的なサイズを決定 2番目の引数は密度
	dBodySetMass(pendulum.body, &(dmass)); // 質量をボディに適用
	dBodySetPosition(pendulum.body, 0.0, 0.0, PENDULUM_Z / 2.0 + STAND_Z / 2.0); // フレームの初期位置
	pendulum.geom = dCreateBox(space, PENDULUM_X, PENDULUM_Y, PENDULUM_Z); // フレームの幾何情報を生成
	dGeomSetBody(pendulum.geom, pendulum.body); // フレームの幾何情報と存在の一致
	
	dBodySetFiniteRotationMode(pendulum.body, 1);
}

// 慣性ロータのスタンドを作成する関数
void InertiaRotorPendulumCreateStand()
{
	dMass dmass;

	stand.body = dBodyCreate(world); // フレームを作成
	dMassSetZero(&(dmass)); // フレームの質量，重心位置，慣性モーメントなどが入っている質量パラメータを初期化します．
	dMassSetBoxTotal(&(dmass), STAND_MASS, STAND_X, STAND_Y, STAND_Z); // フレームの物理的なサイズを決定 2番目の引数は密度
	dBodySetMass(stand.body, &(dmass)); // 質量をボディに適用
	dBodySetPosition(stand.body, 0.0, 0.0, STAND_Z / 2.0); // フレームの初期位置
	stand.geom = dCreateBox(space, STAND_X, STAND_Y, STAND_Z); // フレームの幾何情報を生成
	dGeomSetBody(stand.geom, stand.body); // フレームの幾何情報と存在の一致
}

void InertiaRotorPendulumCreateJointStand()
{
	InertiaRotorPendulumJoint[0] = dJointCreateFixed(world, 0);
	dJointAttach(InertiaRotorPendulumJoint[0], stand.body, 0);
	dJointSetFixed(InertiaRotorPendulumJoint[0]); 
}

void InertiaRotorPendulumCreateJointStandAndPendulum()
{
	InertiaRotorPendulumJoint[1] = dJointCreateHinge(world, 0); // アームと台のジョイントを作成
	dJointAttach(InertiaRotorPendulumJoint[1], stand.body, pendulum.body); // 物体と物体をジョイントで結合
	dJointSetHingeAnchor(InertiaRotorPendulumJoint[1], 0.0, 0.0, STAND_Z / 2.0); // ヒンジの位置を設定, 間接位置xyz
	dJointSetHingeAxis(InertiaRotorPendulumJoint[1], 1.0, 0.0, 0.0); // ヒンジの回転軸(向き)を固定, 軸のxyz
	dJointSetHingeParam(InertiaRotorPendulumJoint[1], dParamHiStop, dInfinity);
    dJointSetHingeParam(InertiaRotorPendulumJoint[1], dParamLoStop,-dInfinity);
}

void InertiaRotorPendulumCreateJointPendulumAndRotor()
{
	InertiaRotorPendulumJoint[2] = dJointCreateHinge(world, 0); // アームと台のジョイントを作成
	dJointAttach(InertiaRotorPendulumJoint[2], rotor.body, pendulum.body); // 物体と物体をジョイントで結合
	dJointSetHingeAnchor(InertiaRotorPendulumJoint[2], 0.0, 0.0, PENDULUM_Z); // ヒンジの位置を設定, 間接位置xyz
	dJointSetHingeAxis(InertiaRotorPendulumJoint[2], 1.0, 0.0, 0.0); // ヒンジの回転軸(向き)を固定, 軸のxyz
	dJointSetHingeParam(InertiaRotorPendulumJoint[2], dParamHiStop, dInfinity);
	dJointSetHingeParam(InertiaRotorPendulumJoint[2], dParamLoStop, -dInfinity);
}

//
// 物理エンジンを使用するための初期化を全てここで行い世界を構築する
//
void InitALL()
{
//	srand(time(NULL));
//	rand();
	setDrawStuff();
	dInitODE2(0);
	dAllocateODEDataForThread(dAllocateMaskAll);
	world = dWorldCreate();                  // ワールドの生成
	space = dHashSpaceCreate(0);          // 衝突用空間の創造
	contactgroup = dJointGroupCreate(0);  // ジョイントグループの生成
	dWorldSetGravity (world, 0.0, 0.0, GRAVITY);       // 重力加速度を決定	
	ground = dCreatePlane(space, 0.0, 0.0, 1.0, 0.0); // 地面（平面ジオメトリ）の生成
}

//
// 車輪とフレームを繋ぐ回転軸を作成する
//
void SingleWheelTypeModelCreateJointFrameAndWheel()
{
	SingleWheelTypeModelJoint[0] = dJointCreateHinge(world, 0); // アームと台のジョイントを作成
	dJointAttach(SingleWheelTypeModelJoint[0], frame.body, wheel.body); // 物体と物体をジョイントで結合
	dJointSetHingeAnchor(SingleWheelTypeModelJoint[0], 0.0, 0.0, WHEEL_RADIUS); // ヒンジの位置を設定, 間接位置xyz
	dJointSetHingeAxis(SingleWheelTypeModelJoint[0], 0.0, 1.0, 0.0); // ヒンジの回転軸(向き)を固定, 軸のxyz
	dJointSetHingeParam(SingleWheelTypeModelJoint[0], dParamHiStop, dInfinity);
	dJointSetHingeParam(SingleWheelTypeModelJoint[0], dParamLoStop, -dInfinity);
}

void MatrixTest(void)
{
	// 行列を作成
	CMatrix M;
	CMatrix N;
	CMatrix L; // 計算の途中に使用する行列
	CMatrix Q;

	// 行列のサイズを指定して0で初期化
	M.Initialize(2, 2, 0.0);
	N.Initialize(2, 2, 0.0);
	L.Initialize(2, 2, 0.0);
	Q.Initialize(3, 3, 0.0);

	// 各要素に値を代入
	M(0, 0) = 5.0; M(0, 1) = 6.0;
	M(1, 0) = 7.0; M(1, 1) = 8.0;

	N(0, 0) = 1.0; N(0, 1) = 2.0;
	N(1, 0) = 3.0; N(1, 1) = 4.0;

	PrintMatrix("matrix M =", M);
	PrintMatrix("matrix N =", N);

	L = M + N; // L = M + N 行列の和
	PrintMatrix("matrix sum M + N = ", L);

	L = M - N; // L = M - N 行列の差
	PrintMatrix("matrix difference M - N = ", L);

	L = M * N; // L = M * N 行列の積
	PrintMatrix("matrix product 1 M * N =", L);

	L = M.PseudoInverse(); // Mの逆行列
	PrintMatrix("matrix inverse M^-1 =", L);

	L = M.PseudoInverse(); // Mの逆行列
	L = L * M; // 逆行列と元の行列を掛けて単位行列に戻るなら正しく逆行列が計算されていることになる
	PrintMatrix("matrix inverse check 1 M^-1 * M =", L);

	L = M.PseudoInverse(); // Mの逆行列
	L = M * L; // 逆行列と元の行列を掛けて単位行列に戻るなら正しく逆行列が計算されていることになる
	PrintMatrix("matrix inverse check 2 M * M^-1 =", L);

	L = M.Transpose(); // Mの転置行列
	PrintMatrix("matrix trancepose M^T =", L);

	double md = M.Determinant();
	printf("matrix determinant det(M) = \n%lf\n", md);

	double tr = M.Trace();
	printf("matrix trace M tr(M) = \n%lf\n", tr);
}

//
// ここからプログラムが開始する
//
int main(int argc, char *argv[])
{
	//
	// 1.使用する制御対象を選ぶ(1重倒立振り子、2重倒立振り子、振り上げ型倒立振り子、1輪車、2輪車)
	//
	cout << "            Controlling Model: Number" << endl;
	cout << "      Single Wheel Type Model: 1" << endl;
	cout << "       Inertia Rotor Pendulum: 2" << endl;
	cout << "What do you select a controlling model?" << endl;
	cout << "Please, input model number >> ";
	cin >> RunFlag;

	//
	// 2.初期化
	//
	InitializeAllParameter();

	//
	// 3.学習、初期値応答、適応の3種類を選択できるようにする(未実装)
	//	
	cout << endl;
	cout << "LearningControl :      1" << endl;
	cout << "InitialValueResponse : 2" << endl;
	cout << "AdaptiveControl :      3" << endl;
	cout << "Select controlling mode >> ";
	cin >> ControlFlag;

	// 学習対象によって初期値応答、適応の初期ゲインを与える
	if(RunFlag == SingleWheelTypeModel)
	{
		switch(ControlFlag)
		{
			case(InitialValueResponse):
				G(0, 0) = -867.349; G(0, 1) = -180.239; G(0, 2) = -247.253; G(0, 3) = -93.7442;
				break;
			case(AdaptiveControl):
				G(0, 0) = -867.349; G(0, 1) = -180.239; G(0, 2) = -247.253; G(0, 3) = -93.7442;
				Lambda1 = 0.3;
				break;
			default:
				break;
		}
	}
	else if(RunFlag == InertiaRotorPendulum)
	{
		switch(ControlFlag)
		{
			case(InitialValueResponse):
				G(0, 0) = -1809.79; G(0, 1) = -545.008; G(0, 2) = 3.593; G(0, 3) = 7.17719;
				break;
			case(AdaptiveControl):
				G(0, 0) = -1809.79; G(0, 1) = -545.008; G(0, 2) = 3.593; G(0, 3) = 7.17719;
				Lambda1 = 0.3;
				break;
		}
	}

	//
	// 4.学習回数を入力
	//
	cout << endl;
	cout << "Input Learning Length (ex, 100, 150) >> ";
	cin >> LearningLengthFinal;

	//
	// 使用する制御対象を作成
	//
	switch(RunFlag)
	{
		case(SingleWheelTypeModel):
			SingleWheelTypeModelCreateWheel();
			SingleWheelTypeModelCreateFrame();
			SingleWheelTypeModelCreateJointFrameAndWheel();
			break;
		case(InertiaRotorPendulum):	
			InertiaRotorPendulumCreateStand();
			InertiaRotorPendulumCreatePendulum();
			InertiaRotorPendulumCreateRotor();
			InertiaRotorPendulumCreateJointStand();
			InertiaRotorPendulumCreateJointStandAndPendulum();
			InertiaRotorPendulumCreateJointPendulumAndRotor();
			break;
		default:
			break;
	}
	

	//
	// シミュレーションループへ入る
	//
	restart(); // あらかじめ対象をリセットしておく	
	dsSimulationLoop(argc,argv, 320, 240, &fn);
	return 0;
}


//
// 衝突検出用コールバック関数(二つの物体(ジオメトリ)が衝突する可能性のある時に呼び出される)
//
static void nearCallback (void *data, dGeomID o1, dGeomID o2)
{
	int i;

	// 二つの物体がどちらも床でない(=どちらも物体同士)なら衝突判定を回避
	if ((o1 != ground) && (o2 != ground))
	{
		return; // 物体同士は衝突検出を行わない
	}

	// 二つの物体が連結されているなら衝突判定を回避
	//exit without doing anything if the two bodies are connected by a joint
	dBodyID b1 = dGeomGetBody(o1);
	dBodyID b2 = dGeomGetBody(o2);
	if (b1 && b2 && dAreConnected (b1, b2))
	{
		return; // 連結されている物体同士は衝突検出を行わない
	}

	dContact contact[MAX_CONTACTS];   // up to MAX_CONTACTS contacts per box-box

	// 衝突フラグを判定して衝突点の数を検出
	int numc = dCollide(o1, o2, MAX_CONTACTS, &contact[0].geom, sizeof(dContact));
	if (numc > 0) {
		for (i=0; i < numc; i++)
		{		
			// 反発係数eを設定するフラグを立てる
			contact[i].surface.mode = dContactBounce;

			// 反発係数eを無くして跳ね返らないように設定
			contact[i].surface.bounce = 0.0;
			contact[i].surface.bounce_vel = 0.0;

			// 摩擦係数μを設定
			contact[i].surface.mu = dInfinity; // 摩擦係数を無限大に指定
			

			// 接触ジョイントを作成して摩擦を適用
			dJointID c = dJointCreateContact(world, contactgroup, &contact[i]);
			dJointAttach (c, dGeomGetBody(contact[i].geom.g1), dGeomGetBody(contact[i].geom.g2));
		}
	}
}

//
// 指定された物体のロー角、ピッチ角、ヨー角およびその角速度を取得する関数
//
void GetRollPitchYaw(dBodyID tmpbody, dReal *rpy, const dReal *angle_rpy)
{
	const dReal *rot = dBodyGetRotation(tmpbody);
	dReal r11,r12,r13,r21,r22,r23,r31,r32,r33;
	dReal pitch, yaw, roll;

	r11 = rot[0];    r12 = rot[1];    r13 = rot[2];
	r21 = rot[4];    r22 = rot[5];    r23 = rot[6];
	r31 = rot[8];    r32 = rot[9];    r33 = rot[10];

	pitch = atan2(-r31, sqrt(r32*r32+r33*r33));
	yaw   = atan2(r21,r11);
	roll  = atan2(r32,r33);

	rpy[0] = roll;
	rpy[1] = pitch;
	rpy[2] = yaw;

	angle_rpy = dBodyGetAngularVel(tmpbody);
}

double PrevRotorAngle;

//
// シミュレーションループ
//
void simLoop(int pause)
{
	// 実世界での現実の時間をタイマーを使って取得
	dStopwatchStop(&mytimer); // 実時間をストップさせる
//	printf("realtime = %f, physics time = %f\n",dStopwatchTime(&mytimer), phystime); //実時間と物理空間の時間
//	drawtime += (dStopwatchTime(&mytimer) - realtime); // 現在時間から過去の時間の差を足し続けて実時間の経過時間とする
	realtime = dStopwatchTime(&mytimer); // 現在時間を新たに取得
	dStopwatchStart(&mytimer); // 実時間を再開させる

	// 物理空間での時間をを実時間に抑える
	if(phystime <= realtime)
	{
		//
		// 制御対象をリセット
		//
		ResetModel();

		//
		// 制御入力決定(雑音と感度計算の準備)
		//
		Control();

		//
		// 入力を制御対象に加える
		//
		switch(RunFlag)
		{
			case(SingleWheelTypeModel):
				dJointAddHingeTorque(SingleWheelTypeModelJoint[0], u(0, 0)); // トルク制御 [Nm]
				break;
			case(InertiaRotorPendulum):
				dJointAddHingeTorque(InertiaRotorPendulumJoint[2], u(0, 0)); // トルク制御 [Nm]
				break;
			default:
				break;
		}
		
		//
		// ODEの世界を更新
		//
		dSpaceCollide(space, 0, &nearCallback); // 衝突検出
		dWorldStep(world, DT); // 微小時間 dt 間隔で世界を更新
		dJointGroupEmpty(contactgroup); // ジョイントグループを空にする
		RunTime += SAMPLINGTIME; // 実行時間を計算
		phystime += DT; // 物理エンジン内の時間を算出
		
		//
		// 対象のモデルから状態を取得する
		//
		switch(RunFlag)
		{
			case(SingleWheelTypeModel):
				//
				// 前クロックの角度を一時的に保存(各角速度の計算に必要)
				//
				prev_frame_pitch = frame_rpy[1];
				prev_wheel_pitch = wheel_rpy[1];
		
				//
				// 新たな角度、角速度を取得
				//
				GetRollPitchYaw(frame.body, frame_rpy, frame_angle_rpy);
				GetRollPitchYaw(wheel.body, wheel_rpy, wheel_angle_rpy);

				//
				// 制御対象の角度、角速度を状態変数に移す
				//
				x(0, 0) = frame_rpy[1]; // フレームの角度
				x(1, 0) = (frame_rpy[1] - prev_frame_pitch) / DT; // フレームの角速度 ODE内で提供されている関数は用いない
				x(2, 0) = wheel_rpy[1]; // ホイールの角度
				x(3, 0) = (wheel_rpy[1] - prev_wheel_pitch) / DT; // ホイールの角速度 ODE内で提供されている関数は用いない
				break;
			case(InertiaRotorPendulum):
				dJointGetHingeAngleInfiniteDiff(InertiaRotorPendulumJoint[2], &InertiaRotorAngle, DT); // 角速度を角度の差分に直して足し上げて角度に直す関数を呼ぶ
				x(0, 0) = dJointGetHingeAngle(InertiaRotorPendulumJoint[1]); // 振り子の角度
				x(1, 0) = dJointGetHingeAngleRate(InertiaRotorPendulumJoint[1]); // 振り子の角速度
				x(2, 0) = InertiaRotorAngle; // ローターの角度 -無限<--->+無限
				x(3, 0) = dJointGetHingeAngleRate(InertiaRotorPendulumJoint[2]); // ローターの角速度
				break;
			default:
				break;
		}

		//
		// 状態データを記録
		//
		RecordGraphx();

		//
		// 感度計算を実行
		//
		SensitivityMain();

		//
		// 現在ブロック長さを更新
		//
		BlockLength++; // ブロック長さを1つ進める

		if(BlockLengthFinal <= BlockLength)
		{ // ブロック長さを超えたら

			//
			// ゲインに関わる更新
			//
			switch(ControlFlag)
			{
				case(LearningControl):
					GainUpdate();
					RecordGraphG();
					break;
				case(InitialValueResponse):	
					RecordGraphG();
					break;
				case(AdaptiveControl):
					GainUpdate();
					RecordGraphG();
					break;
				default:
					break;
			}
			//
			// 現在ブロック長さを初期値1で初期化
			//
			BlockLength = 1; // ブロック長さを元に戻す
			LearningLength++; // 学習回数を1つ進める
			
		}

		//
		// 最終学習回数に到達すれば制御を終了する
		//
		if(LearningLength >= LearningLengthFinal)
		{
			printf("LearningLength %d\n", LearningLength);
			exit(EXIT_SUCCESS);
		}
	}

	// フレームレート(FPS)を管理
	dReal DrawSize[3] = {0.0, 0.0, 0.0};
	/*
	if(drawtime >= SPF)
	{*/
		switch(RunFlag)
		{
			case(SingleWheelTypeModel):
				// 描画
				dsSetColor(1.0, 0.0, 0.0); // 色の設定
				dsDrawCylinderD(dBodyGetPosition(wheel.body), dBodyGetRotation(wheel.body), WHEEL_LENGTH, WHEEL_RADIUS); // 車輪の描画

				dsSetColor(0.0, 1.0, 0.0); // 色の設定
				DrawSize[0] = FRAME_X; DrawSize[1] = FRAME_Y; DrawSize[2] = FRAME_Z;
				dsDrawBox(dBodyGetPosition(frame.body), dBodyGetRotation(frame.body), DrawSize); // アームの描画
				break;
			case(InertiaRotorPendulum):
				dsSetColor(1.0, 0.0, 0.0); // 色の設定
				
				DrawSize[0] = STAND_X; DrawSize[1] = STAND_Y; DrawSize[2] = STAND_Z;
				dsDrawBox(dBodyGetPosition(stand.body), dBodyGetRotation(stand.body), DrawSize); // アームの描画

				dsSetColor(0.0, 1.0, 0.0); // 色の設定
				dsDrawCylinderD(dBodyGetPosition(rotor.body), dBodyGetRotation(rotor.body), ROTOR_LENGTH, ROTOR_RADIUS); // 車輪の描画

				
				dsSetColor(0.0, 0.0, 1.0);
				DrawSize[0] = PENDULUM_X; DrawSize[1] = PENDULUM_Y; DrawSize[2] = PENDULUM_Z;
				dsDrawBox(dBodyGetPosition(pendulum.body), dBodyGetRotation(pendulum.body), DrawSize); // 振り子の描画
				break;
			default:
				break;
		}
		/*
		drawtime = 0.0;
	}	*/
}


//
// 物理シミュレーションが開始する前の処理関数
//
void start()
{
  dAllocateODEDataForThread(dAllocateMaskAll);

  static float xyz[3] = {2.0, 0.0, 0.4};  // 視点の位置
  static float hpr[3] = {180.0, 0.0, 0.0};  // 視線の方向
  
  dsSetViewpoint(xyz,hpr);  // カメラの設定

  //Timerを実装
  dStopwatchReset(&mytimer);
  dStopwatchStart(&mytimer);
}

//
// 停止時実行関数
//
void stop()
{
  printf("Stop timer\n");
  dStopwatchStop(&mytimer);
  printf("time=%18.15le\n",dStopwatchTime(&mytimer));
}

//
// キー入力で呼び出される関数
//
void command(int cmd)
{
	switch (cmd)
	{
	//case 'r':
	//	restart();
	//	break;
	default:
		break;
	}
}

//
// 物理エンジンのシミュレーションをリセットする関数
//
void restart()
{
	int i;

	// 接触点のジョイントの破壊
	dJointGroupDestroy(contactgroup);

	// 回転軸、固定軸のジョイントの破壊
	switch(RunFlag)
	{
		case(SingleWheelTypeModel):
			for(i = SingleWheelTypeModelJointMax-1; i >= 0; i--)
			{
				dJointDestroy(SingleWheelTypeModelJoint[i]);
			}
			break;
		case(InertiaRotorPendulum):
			for(i = InertiaRotorPendulumJointMax-1; i >= 0; i--)
			{
				dJointDestroy(InertiaRotorPendulumJoint[i]);
			}
			break;
		default:
			break;
	}
	
	// 剛体モデルと衝突検出に使用する幾何情報の破壊
	switch(RunFlag)
	{
		case(SingleWheelTypeModel):
			dBodyDestroy(wheel.body);
			dGeomDestroy(wheel.geom);
			dBodyDestroy(frame.body);
			dGeomDestroy(frame.geom);
			break;
		case(InertiaRotorPendulum):
			dBodyDestroy(rotor.body);
			dGeomDestroy(rotor.geom);
			dBodyDestroy(pendulum.body);
			dGeomDestroy(pendulum.geom);
			dBodyDestroy(stand.body);
			dGeomDestroy(stand.geom);
			break;
		default:
			break;
	}
	
	// 破壊した全てを創造
	contactgroup = dJointGroupCreate(0);
	switch(RunFlag)
	{
		case(SingleWheelTypeModel):
			SingleWheelTypeModelCreateWheel();
			SingleWheelTypeModelCreateFrame();
			SingleWheelTypeModelCreateJointFrameAndWheel();
			break;
		case(InertiaRotorPendulum):
			InertiaRotorPendulumCreateStand();
			InertiaRotorPendulumCreatePendulum();
			InertiaRotorPendulumCreateRotor();
			InertiaRotorPendulumCreateJointStand();
			InertiaRotorPendulumCreateJointStandAndPendulum();
			InertiaRotorPendulumCreateJointPendulumAndRotor();
			break;
		default:
			break;
	}
/*
	// 念のためシミュレーションをスキップする 3～5程度
	for(i=0; i < 3; i++)
	{
		dSpaceCollide(space, 0, &nearCallback); // 地面との衝突を考慮
		dWorldStep(world, DT); // シミュレーションステップを進める
		phystime += DT;
		dJointGroupEmpty(contactgroup); // 地面との衝突を初期化
		if(RunFlag == SingleWheelTypeModel)
		{
			switch(ControlFlag)
			{
				case(InitialValueResponse):
					//dJointSetHingeParam (SingleWheelTypeModelJoint[0], dParamFMax, 60);
					//dJointSetHingeParam(SingleWheelTypeModelJoint[0], dParamVel, M_PI / 3.0); // 少しずらす
					dJointAddHingeTorque(SingleWheelTypeModelJoint[0], 0.001);
					break;
				case(AdaptiveControl):
					//dJointSetHingeParam (SingleWheelTypeModelJoint[0], dParamFMax, 60);
					//dJointSetHingeParam(SingleWheelTypeModelJoint[0], dParamVel, 0.1); // 少しずらす
					break;
			}
		}
		if(RunFlag == InertiaRotorPendulum)
		{
			switch(ControlFlag)
			{
				case(InitialValueResponse):
					//dJointSetHingeParam (InertiaRotorPendulumJoint[2], dParamFMax, 123);
					//dJointSetHingeParam(InertiaRotorPendulumJoint[2], dParamVel, -M_PI * 1000.0); // 少しずらす
					break;
				case(AdaptiveControl):
					//dJointSetHingeParam (InertiaRotorPendulumJoint[2], dParamFMax, 10);
					//dJointSetHingeParam(InertiaRotorPendulumJoint[2], dParamVel, 1000); // 少しずらす
					break;
			}
		}
	}
*/
	
}

//
// 描画を受け持つ関数の設定
//
void setDrawStuff()
{
	fn.version = DS_VERSION;    // ドロースタッフのバージョン
	fn.start   = &start;        // 前処理 start関数のポインタ
	fn.step    = &simLoop;      // simLoop関数のポインタ
	fn.stop    = &stop;         // 停止時に呼ばれる関数ポインタ
	fn.command = &command;      // キー入力処理の関数ポインタ
	fn.path_to_textures = "../../drawstuff/textures"; // テクスチャファイルの相対パス
}

void dJointGetHingeAngleInfiniteDiff(dJointID jointID, double *Angle, double dt)
{
	*Angle += (dJointGetHingeAngleRate(jointID) * (dt));
}